<?php

namespace Plugins\StripeGateway;

use App\Models\Invoice;
use App\Models\Plugin;
use App\Models\Transaction;
use App\Services\Billing\BillingService;
use App\Services\Payments\PaymentGatewayInterface;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class StripeGateway implements PaymentGatewayInterface
{
    public function __construct(private ?Plugin $plugin = null)
    {
    }

    public function slug(): string
    {
        return 'stripe';
    }

    public function displayName(): string
    {
        return 'Stripe';
    }

    public function isConfigured(): bool
    {
        $key = trim((string) ($this->plugin?->setting('secret_key') ?? env('STRIPE_SECRET_KEY')));
        return $key !== '';
    }

    public function checkoutFields(): array
    {
        // Stripe Checkout does not require additional user fields.
        return [];
    }

    public function validationRules(): array
    {
        return [];
    }

    public function initiate(Invoice $invoice, array $data): array
    {
        if ($invoice->isPaid()) {
            return ['success' => true, 'message' => 'Invoice already paid.'];
        }

        if (!$this->isConfigured()) {
            return ['success' => false, 'message' => 'Stripe gateway is not configured.'];
        }

        $secret = trim((string) ($this->plugin?->setting('secret_key') ?? env('STRIPE_SECRET_KEY')));

        // Ensure payment token exists (used to validate return)
        $meta = $invoice->meta ?? [];
        if (empty($meta['payment_token'])) {
            $meta['payment_token'] = Str::random(32);
            $invoice->meta = $meta;
            $invoice->save();
        }

        $successUrl = route('payments.return', [
            'gateway' => $this->slug(),
            'invoice' => $invoice->id,
            'ptoken' => $meta['payment_token'],
        ], true);

        $joiner = str_contains($successUrl, '?') ? '&' : '?';
        $successUrl .= $joiner . 'session_id={CHECKOUT_SESSION_ID}';

        $cancelUrl = route('payments.cancel', [
            'gateway' => $this->slug(),
            'invoice' => $invoice->id,
        ], true);

        $currency = strtolower($invoice->currency ?: 'pkr');
        $amount = (int) round(((float) $invoice->total) * 100);

        $params = [
            'mode' => 'payment',
            'success_url' => $successUrl,
            'cancel_url' => $cancelUrl,
            'client_reference_id' => (string) $invoice->id,
            'metadata[invoice_id]' => (string) $invoice->id,
            'payment_method_types[0]' => 'card',
            'line_items[0][quantity]' => 1,
            'line_items[0][price_data][currency]' => $currency,
            'line_items[0][price_data][unit_amount]' => $amount,
            'line_items[0][price_data][product_data][name]' => 'Invoice ' . $invoice->number,
        ];

        if ($invoice->user && $invoice->user->email) {
            $params['customer_email'] = $invoice->user->email;
        }

        $transaction = Transaction::query()->create([
            'invoice_id' => $invoice->id,
            'user_id' => $invoice->user_id,
            'gateway' => $this->slug(),
            'amount' => (float) $invoice->total,
            'currency' => $invoice->currency,
            'status' => 'initiated',
            'raw_request' => $params,
        ]);

        try {
            $resp = Http::asForm()
                ->timeout(45)
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $secret,
                ])
                ->post('https://api.stripe.com/v1/checkout/sessions', $params);

            $json = $resp->json() ?: [];

            if (!$resp->successful()) {
                $msg = (string) (($json['error']['message'] ?? '') ?: 'Stripe API request failed.');
                $transaction->update([
                    'status' => 'failed',
                    'response_message' => $msg,
                    'raw_response' => $json,
                ]);
                return ['success' => false, 'message' => $msg];
            }

            $sessionId = (string) ($json['id'] ?? '');
            $url = (string) ($json['url'] ?? '');

            $transaction->update([
                'txn_ref' => $sessionId !== '' ? $sessionId : null,
                'raw_response' => $json,
            ]);

            if ($url === '') {
                return ['success' => false, 'message' => 'Stripe session created but no redirect URL returned.'];
            }

            return [
                'success' => true,
                'redirect_url' => $url,
                'message' => 'Redirecting to Stripe...',
            ];
        } catch (\Throwable $e) {
            $transaction->update([
                'status' => 'error',
                'response_message' => $e->getMessage(),
            ]);

            return ['success' => false, 'message' => 'Stripe error: ' . $e->getMessage()];
        }
    }

    public function handleReturn(Request $request): array
    {
        if (!$this->isConfigured()) {
            return ['success' => false, 'message' => 'Stripe gateway is not configured.'];
        }

        $invoiceId = (int) $request->query('invoice');
        $token = (string) $request->query('ptoken');
        $sessionId = (string) $request->query('session_id');

        if ($invoiceId <= 0 || $token === '' || $sessionId === '') {
            return ['success' => false, 'message' => 'Invalid return parameters.'];
        }

        $invoice = Invoice::query()->find($invoiceId);
        if (!$invoice) {
            return ['success' => false, 'message' => 'Invoice not found.'];
        }

        $meta = $invoice->meta ?? [];
        if (empty($meta['payment_token']) || !hash_equals((string) $meta['payment_token'], $token)) {
            return ['success' => false, 'message' => 'Invalid payment token.'];
        }

        if ($invoice->isPaid()) {
            return ['success' => true, 'invoice_id' => $invoice->id, 'message' => 'Invoice already paid.'];
        }

        $secret = trim((string) ($this->plugin?->setting('secret_key') ?? env('STRIPE_SECRET_KEY')));

        try {
            $resp = Http::timeout(45)
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $secret,
                ])
                ->get('https://api.stripe.com/v1/checkout/sessions/' . urlencode($sessionId));

            $json = $resp->json() ?: [];
            $paymentStatus = (string) ($json['payment_status'] ?? '');
            $status = (string) ($json['status'] ?? '');

            $transaction = Transaction::query()->where('invoice_id', $invoice->id)
                ->where('gateway', $this->slug())
                ->where('txn_ref', $sessionId)
                ->latest()
                ->first();

            if (!$transaction) {
                $transaction = Transaction::query()->create([
                    'invoice_id' => $invoice->id,
                    'user_id' => $invoice->user_id,
                    'gateway' => $this->slug(),
                    'txn_ref' => $sessionId,
                    'amount' => (float) $invoice->total,
                    'currency' => $invoice->currency,
                    'status' => 'initiated',
                ]);
            }

            $transaction->update([
                'raw_response' => $json,
            ]);

            if (!$resp->successful()) {
                $msg = (string) (($json['error']['message'] ?? '') ?: 'Stripe verification failed.');
                $transaction->update([
                    'status' => 'failed',
                    'response_message' => $msg,
                ]);
                return ['success' => false, 'invoice_id' => $invoice->id, 'message' => $msg];
            }

            if ($paymentStatus === 'paid') {
                $transaction->update([
                    'status' => 'success',
                    'response_message' => 'paid',
                ]);

                app(BillingService::class)->markInvoicePaid($invoice, $transaction, [
                    'gateway' => $this->slug(),
                    'stripe_status' => $status,
                ]);

                return ['success' => true, 'invoice_id' => $invoice->id, 'message' => 'Payment successful.'];
            }

            $msg = 'Stripe session is not paid yet (status: ' . $paymentStatus . ').';
            $transaction->update([
                'status' => 'failed',
                'response_message' => $msg,
            ]);

            return ['success' => false, 'invoice_id' => $invoice->id, 'message' => $msg];
        } catch (\Throwable $e) {
            return ['success' => false, 'invoice_id' => $invoiceId, 'message' => 'Stripe error: ' . $e->getMessage()];
        }
    }

    public function handleWebhook(Request $request): array
    {
        // Optional: implement Stripe webhook verification + invoice marking.
        return ['success' => true, 'message' => 'Webhook not used in this setup.'];
    }
}
