<?php

namespace Plugins\PayPalGateway;

use App\Models\Invoice;
use App\Models\Plugin;
use App\Models\Transaction;
use App\Services\Billing\BillingService;
use App\Services\Payments\PaymentGatewayInterface;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class PayPalGateway implements PaymentGatewayInterface
{
    private ?string $accessToken = null;

    public function __construct(private ?Plugin $plugin = null)
    {
    }

    public function slug(): string
    {
        return 'paypal';
    }

    public function displayName(): string
    {
        return 'PayPal';
    }

    public function isConfigured(): bool
    {
        return trim((string) ($this->plugin?->setting('client_id') ?? env('PAYPAL_CLIENT_ID'))) !== ''
            && trim((string) ($this->plugin?->setting('client_secret') ?? env('PAYPAL_CLIENT_SECRET'))) !== '';
    }

    public function checkoutFields(): array
    {
        return [];
    }

    public function validationRules(): array
    {
        return [];
    }

    public function initiate(Invoice $invoice, array $data): array
    {
        if ($invoice->isPaid()) {
            return ['success' => true, 'message' => 'Invoice already paid.'];
        }

        if (!$this->isConfigured()) {
            return ['success' => false, 'message' => 'PayPal gateway is not configured.'];
        }

        // Ensure payment token exists
        $meta = $invoice->meta ?? [];
        if (empty($meta['payment_token'])) {
            $meta['payment_token'] = Str::random(32);
            $invoice->meta = $meta;
            $invoice->save();
        }

        $returnUrl = route('payments.return', [
            'gateway' => $this->slug(),
            'invoice' => $invoice->id,
            'ptoken' => $meta['payment_token'],
        ], true);

        $cancelUrl = route('payments.cancel', [
            'gateway' => $this->slug(),
            'invoice' => $invoice->id,
        ], true);

        $currency = strtoupper((string) ($this->plugin?->setting('currency_code') ?? env('PAYPAL_CURRENCY', 'USD')));
        $amountValue = number_format((float) $invoice->total, 2, '.', '');

        $transaction = Transaction::query()->create([
            'invoice_id' => $invoice->id,
            'user_id' => $invoice->user_id,
            'gateway' => $this->slug(),
            'amount' => (float) $invoice->total,
            'currency' => $invoice->currency,
            'status' => 'initiated',
        ]);

        try {
            $token = $this->getAccessToken();

            $body = [
                'intent' => 'CAPTURE',
                'purchase_units' => [
                    [
                        'reference_id' => 'invoice_' . $invoice->id,
                        'description' => 'Invoice ' . $invoice->number,
                        'amount' => [
                            'currency_code' => $currency,
                            'value' => $amountValue,
                        ],
                    ],
                ],
                'application_context' => [
                    'return_url' => $returnUrl,
                    'cancel_url' => $cancelUrl,
                    'user_action' => 'PAY_NOW',
                    'brand_name' => config('app.name', 'ClubHoster'),
                ],
            ];

            $transaction->update(['raw_request' => $body]);

            $resp = Http::timeout(45)
                ->withToken($token)
                ->withHeaders(['Content-Type' => 'application/json'])
                ->post($this->baseUrl() . '/v2/checkout/orders', $body);

            $json = $resp->json() ?: [];
            $transaction->update(['raw_response' => $json]);

            if (!$resp->successful()) {
                $msg = (string) (($json['message'] ?? '') ?: 'PayPal create order failed.');
                $transaction->update(['status' => 'failed', 'response_message' => $msg]);
                return ['success' => false, 'message' => $msg];
            }

            $orderId = (string) ($json['id'] ?? '');
            $approveUrl = '';

            foreach (($json['links'] ?? []) as $link) {
                if (($link['rel'] ?? '') === 'approve') {
                    $approveUrl = (string) ($link['href'] ?? '');
                    break;
                }
            }

            $transaction->update([
                'txn_ref' => $orderId !== '' ? $orderId : null,
            ]);

            if ($orderId === '' || $approveUrl === '') {
                $msg = 'PayPal order created but approval URL missing.';
                $transaction->update(['status' => 'failed', 'response_message' => $msg]);
                return ['success' => false, 'message' => $msg];
            }

            return [
                'success' => true,
                'redirect_url' => $approveUrl,
                'message' => 'Redirecting to PayPal...',
            ];
        } catch (\Throwable $e) {
            $transaction->update(['status' => 'error', 'response_message' => $e->getMessage()]);
            return ['success' => false, 'message' => 'PayPal error: ' . $e->getMessage()];
        }
    }

    public function handleReturn(Request $request): array
    {
        if (!$this->isConfigured()) {
            return ['success' => false, 'message' => 'PayPal gateway is not configured.'];
        }

        $invoiceId = (int) $request->query('invoice');
        $ptoken = (string) $request->query('ptoken');

        // PayPal returns order id in `token`
        $orderId = (string) ($request->query('token') ?? $request->query('orderId') ?? $request->query('order_id') ?? '');

        if ($invoiceId <= 0 || $ptoken === '' || $orderId === '') {
            return ['success' => false, 'message' => 'Invalid return parameters.'];
        }

        $invoice = Invoice::query()->find($invoiceId);
        if (!$invoice) {
            return ['success' => false, 'message' => 'Invoice not found.'];
        }

        $meta = $invoice->meta ?? [];
        if (empty($meta['payment_token']) || !hash_equals((string) $meta['payment_token'], $ptoken)) {
            return ['success' => false, 'message' => 'Invalid payment token.'];
        }

        if ($invoice->isPaid()) {
            return ['success' => true, 'invoice_id' => $invoice->id, 'message' => 'Invoice already paid.'];
        }

        $transaction = Transaction::query()->where('invoice_id', $invoice->id)
            ->where('gateway', $this->slug())
            ->where('txn_ref', $orderId)
            ->latest()
            ->first();

        if (!$transaction) {
            $transaction = Transaction::query()->create([
                'invoice_id' => $invoice->id,
                'user_id' => $invoice->user_id,
                'gateway' => $this->slug(),
                'txn_ref' => $orderId,
                'amount' => (float) $invoice->total,
                'currency' => $invoice->currency,
                'status' => 'initiated',
            ]);
        }

        try {
            $tokenApi = $this->getAccessToken();

            $resp = Http::timeout(45)
                ->withToken($tokenApi)
                ->withHeaders(['Content-Type' => 'application/json'])
                ->post($this->baseUrl() . '/v2/checkout/orders/' . urlencode($orderId) . '/capture', []);

            $json = $resp->json() ?: [];
            $transaction->update(['raw_response' => $json]);

            if (!$resp->successful()) {
                $msg = (string) (($json['message'] ?? '') ?: 'PayPal capture failed.');
                $transaction->update(['status' => 'failed', 'response_message' => $msg]);
                return ['success' => false, 'invoice_id' => $invoice->id, 'message' => $msg];
            }

            $status = (string) ($json['status'] ?? '');

            if ($status === 'COMPLETED') {
                $transaction->update(['status' => 'success', 'response_message' => 'COMPLETED']);
                app(BillingService::class)->markInvoicePaid($invoice, $transaction, [
                    'gateway' => $this->slug(),
                    'paypal_status' => $status,
                ]);

                return ['success' => true, 'invoice_id' => $invoice->id, 'message' => 'Payment successful.'];
            }

            $msg = 'PayPal capture status: ' . ($status ?: 'unknown') . '.';
            $transaction->update(['status' => 'failed', 'response_message' => $msg]);

            return ['success' => false, 'invoice_id' => $invoice->id, 'message' => $msg];
        } catch (\Throwable $e) {
            $transaction->update(['status' => 'error', 'response_message' => $e->getMessage()]);
            return ['success' => false, 'invoice_id' => $invoice->id, 'message' => 'PayPal error: ' . $e->getMessage()];
        }
    }

    public function handleWebhook(Request $request): array
    {
        // Optional: implement PayPal webhooks verification + invoice marking.
        return ['success' => true, 'message' => 'Webhook not used in this setup.'];
    }

    private function getAccessToken(): string
    {
        if ($this->accessToken) {
            return $this->accessToken;
        }

        $clientId = trim((string) ($this->plugin?->setting('client_id') ?? env('PAYPAL_CLIENT_ID')));
        $secret = trim((string) ($this->plugin?->setting('client_secret') ?? env('PAYPAL_CLIENT_SECRET')));

        $resp = Http::asForm()
            ->timeout(45)
            ->withBasicAuth($clientId, $secret)
            ->post($this->baseUrl() . '/v1/oauth2/token', [
                'grant_type' => 'client_credentials',
            ]);

        $json = $resp->json() ?: [];
        if (!$resp->successful()) {
            $msg = (string) (($json['error_description'] ?? $json['message'] ?? '') ?: 'Failed to get PayPal access token.');
            throw new \RuntimeException($msg);
        }

        $token = (string) ($json['access_token'] ?? '');
        if ($token === '') {
            throw new \RuntimeException('PayPal access token missing.');
        }

        $this->accessToken = $token;
        return $this->accessToken;
    }

    private function baseUrl(): string
    {
        $mode = (string) ($this->plugin?->setting('mode') ?? env('PAYPAL_MODE', 'sandbox'));
        return $mode === 'live' ? 'https://api-m.paypal.com' : 'https://api-m.sandbox.paypal.com';
    }
}
