<?php

namespace Plugins\NameSiloRegistrar;

use App\Models\Domain;
use App\Models\Plugin;
use Illuminate\Support\Facades\Http;

class NameSiloRegistrar
{
    public function __construct(private ?Plugin $plugin = null)
    {
    }

    public function isConfigured(): bool
    {
        return trim((string) ($this->plugin?->setting('api_key') ?? env('NAMESILO_API_KEY'))) !== '';
    }

    public function handleInvoicePaid(array $payload): void
    {
        if (!$this->isConfigured()) {
            return;
        }

        $invoice = $payload['invoice'] ?? null;
        $order = $payload['order'] ?? null;
        if (!$invoice || !$order) {
            return;
        }

        // Find all domain order items and register those that are pending.
        $orderItems = method_exists($order, 'items') ? $order->items()->where('item_type', 'domain')->get() : collect();

        foreach ($orderItems as $item) {
            $domainId = (int) ($item->reference_id ?? 0);
            if ($domainId <= 0) {
                continue;
            }

            $domain = Domain::query()->find($domainId);
            if (!$domain) {
                continue;
            }

            $option = (string) (($domain->meta['option'] ?? '') ?: 'register');
            if ($option !== 'register') {
                continue;
            }

            if ((string) $domain->status !== 'pending_registration') {
                continue;
            }

            $years = (int) ($domain->years ?: ($item->meta['years'] ?? 1));
            $years = max(1, min(10, $years));

            $res = $this->registerDomain($domain->domain, $years);
            if (!$res['success']) {
                logger()->warning('NameSilo domain registration failed', [
                    'domain' => $domain->domain,
                    'message' => $res['message'],
                    'reply' => $res['reply'] ?? null,
                ]);
                continue;
            }

            // Optionally set name servers
            $nsCsv = trim((string) ($this->plugin?->setting('nameservers') ?? env('NAMESILO_NAMESERVERS', '')));
            if ($nsCsv !== '') {
                $nameservers = array_values(array_filter(array_map('trim', explode(',', $nsCsv))));
                if (count($nameservers) >= 2) {
                    $this->changeNameServers($domain->domain, $nameservers);
                }
            }

            $domain->update([
                'status' => 'active',
                'registrar' => 'namesilo',
                'expires_at' => now()->addYears($years),
                'meta' => array_merge($domain->meta ?? [], [
                    'registered_via' => 'namesilo',
                    'namesilo_reply' => $res['reply'] ?? null,
                ]),
            ]);
        }
    }

    public function registerDomain(string $domain, int $years = 1): array
    {
        $domain = strtolower(trim($domain));
        if ($domain === '') {
            return ['success' => false, 'message' => 'Domain is empty.'];
        }

        // NOTE: NameSilo registerDomain requires account contact defaults on NameSilo.
        // Some accounts may require explicit contact parameters. Extend this plugin as needed.
        $params = [
            'domain' => $domain,
            'years' => $years,
        ];

        return $this->request('registerDomain', $params);
    }

    public function changeNameServers(string $domain, array $nameservers): array
    {
        $domain = strtolower(trim($domain));
        $nameservers = array_values(array_filter(array_map('trim', $nameservers)));
        if ($domain === '' || count($nameservers) < 2) {
            return ['success' => false, 'message' => 'Domain or nameservers invalid.'];
        }

        // NameSilo supports ns1..ns13 parameters in some clients; we map the first N.
        $params = ['domain' => $domain];
        foreach ($nameservers as $i => $ns) {
            $params['ns' . ($i + 1)] = $ns;
        }

        return $this->request('changeNameServers', $params);
    }

    private function request(string $operation, array $params): array
    {
        $key = trim((string) ($this->plugin?->setting('api_key') ?? env('NAMESILO_API_KEY')));
        $version = trim((string) ($this->plugin?->setting('version') ?? env('NAMESILO_VERSION', '1')));

        $query = array_merge([
            'version' => $version,
            'type' => 'json',
            'key' => $key,
        ], $params);

        $url = 'https://www.namesilo.com/api/' . ltrim($operation, '/');

        try {
            $resp = Http::timeout(45)->get($url, $query);
            $json = $resp->json() ?: [];

            $reply = $this->extractReply($json);
            $code = (int) ($reply['code'] ?? 0);
            $detail = (string) ($reply['detail'] ?? '');

            // Common NameSilo success code is 300.
            $success = $resp->successful() && $code === 300;

            return [
                'success' => $success,
                'message' => $success ? 'OK' : ($detail ?: ('NameSilo error (code ' . $code . ')')),
                'reply' => $reply,
                'raw' => $json,
                'http_status' => $resp->status(),
            ];
        } catch (\Throwable $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    private function extractReply(array $json): array
    {
        // NameSilo responses often look like: { "namesilo": { "reply": {...} } }
        if (isset($json['namesilo']['reply']) && is_array($json['namesilo']['reply'])) {
            return $json['namesilo']['reply'];
        }
        if (isset($json['reply']) && is_array($json['reply'])) {
            return $json['reply'];
        }
        return [];
    }
}
