<?php

namespace Plugins\JazzCashGateway;

use App\Models\Invoice;
use App\Models\Plugin;
use App\Models\Transaction;
use App\Services\Billing\BillingService;
use App\Services\Payments\JazzCashService;
use App\Services\Payments\PaymentGatewayInterface;
use Illuminate\Http\Request;

class JazzCashGateway implements PaymentGatewayInterface
{
    public function __construct(private ?Plugin $plugin = null)
    {
    }

    public function slug(): string
    {
        return 'jazzcash';
    }

    public function displayName(): string
    {
        return 'JazzCash';
    }

    public function isConfigured(): bool
    {
        $mid = trim((string) ($this->plugin?->setting('merchant_id') ?? config('jazzcash.merchant_id')));
        $pw = trim((string) ($this->plugin?->setting('password') ?? config('jazzcash.password')));
        $salt = trim((string) ($this->plugin?->setting('integrity_salt') ?? config('jazzcash.integrity_salt')));

        return $mid !== '' && $pw !== '' && $salt !== '';
    }

    public function checkoutFields(): array
    {
        return [
            [
                'key' => 'jazzcash_mobile',
                'label' => 'JazzCash Mobile Number',
                'type' => 'text',
                'placeholder' => '03xxxxxxxxx',
                'help' => 'Used to process JazzCash MWallet transaction.',
            ],
            [
                'key' => 'jazzcash_cnic',
                'label' => 'CNIC (last 6 digits)',
                'type' => 'text',
                'placeholder' => '123456',
            ],
        ];
    }

    public function validationRules(): array
    {
        return [
            'jazzcash_mobile' => ['required', 'string', 'min:10', 'max:20'],
            'jazzcash_cnic' => ['required', 'string', 'min:6', 'max:13'],
        ];
    }

    public function initiate(Invoice $invoice, array $data): array
    {
        if ($invoice->isPaid()) {
            return ['success' => true, 'message' => 'Invoice already paid.'];
        }

        if (!$this->isConfigured()) {
            return ['success' => false, 'message' => 'JazzCash gateway is not configured.'];
        }

        $merchantId = trim((string) ($this->plugin?->setting('merchant_id') ?? config('jazzcash.merchant_id')));
        $password = trim((string) ($this->plugin?->setting('password') ?? config('jazzcash.password')));
        $salt = trim((string) ($this->plugin?->setting('integrity_salt') ?? config('jazzcash.integrity_salt')));
        $environment = (string) ($this->plugin?->setting('environment') ?? config('jazzcash.environment', 'sandbox'));

        $payer = [
            'mobile' => $data['jazzcash_mobile'] ?? null,
            'cnic' => $data['jazzcash_cnic'] ?? null,
        ];

        $service = app(JazzCashService::class);
        $txnRef = 'JC' . now()->format('YmdHis') . random_int(100, 999);

        $result = $service->doMWalletTransaction($invoice, $payer, [
            'merchant_id' => $merchantId,
            'password' => $password,
            'integrity_salt' => $salt,
            'environment' => $environment,
            'txn_ref' => $txnRef,
        ]);

        $transaction = Transaction::query()->create([
            'invoice_id' => $invoice->id,
            'user_id' => $invoice->user_id,
            'gateway' => $this->slug(),
            'txn_ref' => $result['txn_ref'] ?? $txnRef,
            'amount' => (float) $invoice->total,
            'currency' => $invoice->currency,
            'status' => ($result['success'] ?? false) ? 'success' : (($result['response_code'] ?? null) ? 'failed' : 'error'),
            'response_code' => $result['response_code'] ?? null,
            'response_message' => $result['message'] ?? null,
            'raw_request' => $result['payload'] ?? null,
            'raw_response' => $result['response'] ?? null,
        ]);

        if (($result['success'] ?? false) === true) {
            app(BillingService::class)->markInvoicePaid($invoice, $transaction, [
                'gateway' => $this->slug(),
            ]);

            return ['success' => true, 'message' => $result['message'] ?? 'Payment successful.'];
        }

        return ['success' => false, 'message' => $result['message'] ?? 'Payment failed.'];
    }

    public function handleReturn(Request $request): array
    {
        return ['success' => false, 'message' => 'JazzCash does not use a return URL in this flow.'];
    }

    public function handleWebhook(Request $request): array
    {
        return ['success' => false, 'message' => 'JazzCash webhook is not implemented.'];
    }
}
