<?php

namespace Plugins\CpanelAutomation;

use App\Models\Plugin;
use App\Models\Service;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class WhmProvisioner
{
    public function __construct(private ?Plugin $plugin = null)
    {
    }

    public function isConfigured(): bool
    {
        $host = trim((string) ($this->plugin?->setting('whm_host') ?? env('WHM_HOST')));
        $username = trim((string) ($this->plugin?->setting('whm_username') ?? env('WHM_USERNAME')));
        $token = trim((string) ($this->plugin?->setting('whm_token') ?? env('WHM_API_TOKEN')));
        $pkg = trim((string) ($this->plugin?->setting('default_package') ?? env('WHM_DEFAULT_PACKAGE')));

        return $host !== '' && $username !== '' && $token !== '' && $pkg !== '';
    }

    public function handleInvoicePaid(array $payload): void
    {
        if (!$this->isConfigured()) {
            return;
        }

        $order = $payload['order'] ?? null;
        if (!$order || !isset($order->id)) {
            return;
        }

        // Provision all hosting services on the order
        $services = Service::query()
            ->where('order_id', $order->id)
            ->where('type', 'hosting')
            ->get();

        foreach ($services as $service) {
            $meta = $service->meta ?? [];
            if (!empty($meta['whm_provisioned'])) {
                continue;
            }

            $domain = (string) ($meta['domain'] ?? '');
            if ($domain === '') {
                // If domain is linked via domain_id, you can extend here.
                continue;
            }

            $cpUser = (string) ($meta['cpanel_username'] ?? '');
            $cpPass = (string) ($meta['cpanel_password'] ?? '');

            if ($cpUser === '') {
                $cpUser = $this->generateCpanelUsername($domain);
            }
            if ($cpPass === '') {
                $cpPass = $this->generatePassword();
            }

            $plan = trim((string) ($this->plugin?->setting('default_package') ?? env('WHM_DEFAULT_PACKAGE')));

            $res = $this->createAccount(
                domain: $domain,
                username: $cpUser,
                password: $cpPass,
                plan: $plan,
                contactEmail: $service->user?->email
            );

            if (!$res['success']) {
                $service->update([
                    'meta' => array_merge($meta, [
                        'whm_provisioned' => false,
                        'whm_error' => $res['message'],
                        'whm_last_reply' => $res['raw'] ?? null,
                    ]),
                ]);
                continue;
            }

            $service->update([
                'status' => 'active',
                'meta' => array_merge($meta, [
                    'cpanel_username' => $cpUser,
                    'cpanel_password' => $cpPass,
                    'whm_provisioned' => true,
                    'whm_last_reply' => $res['raw'] ?? null,
                    'provisioned_at' => now()->toDateTimeString(),
                ]),
            ]);
        }
    }

    public function createAccount(string $domain, string $username, string $password, string $plan, ?string $contactEmail = null): array
    {
        $host = trim((string) ($this->plugin?->setting('whm_host') ?? env('WHM_HOST')));
        $port = (int) ($this->plugin?->setting('whm_port') ?? env('WHM_PORT', 2087));
        $whmUser = trim((string) ($this->plugin?->setting('whm_username') ?? env('WHM_USERNAME')));
        $token = trim((string) ($this->plugin?->setting('whm_token') ?? env('WHM_API_TOKEN')));
        $verifyTls = (string) ($this->plugin?->setting('verify_tls') ?? env('WHM_VERIFY_TLS', '1'));

        $verify = $verifyTls !== '0';

        $url = 'https://' . $host . ':' . $port . '/json-api/createacct';

        $query = [
            'api.version' => 1,
            'username' => $username,
            'domain' => $domain,
            'plan' => $plan,
            'password' => $password,
        ];

        if ($contactEmail) {
            $query['contactemail'] = $contactEmail;
        }

        try {
            $resp = Http::timeout(60)
                ->withOptions(['verify' => $verify])
                ->withHeaders([
                    'Authorization' => 'whm ' . $whmUser . ':' . $token,
                ])
                ->get($url, $query);

            $json = $resp->json() ?: [];
            $result = (int) (data_get($json, 'metadata.result') ?? data_get($json, 'result', 0));
            $reason = (string) (data_get($json, 'metadata.reason') ?? data_get($json, 'reason', ''));

            $success = $resp->successful() && $result === 1;

            return [
                'success' => $success,
                'message' => $success ? 'OK' : ($reason ?: 'WHM createacct failed.'),
                'raw' => $json,
                'http_status' => $resp->status(),
            ];
        } catch (\Throwable $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    private function generateCpanelUsername(string $domain): string
    {
        $base = strtolower(preg_replace('/[^a-z0-9]/i', '', explode('.', $domain)[0] ?? 'user'));
        $base = $base ?: 'user';

        // Conservative: keep within 8 chars for broad cPanel compatibility.
        $base = substr($base, 0, 5);
        $suffix = substr((string) random_int(100, 999), 0, 3);

        return substr($base . $suffix, 0, 8);
    }

    private function generatePassword(): string
    {
        // cPanel password rules vary; generate a reasonably complex password.
        return Str::random(18) . 'A1!';
    }
}
