<?php

namespace Database\Seeders;

use App\Models\Plan;
use App\Models\TldPricing;
use App\Models\User;
use App\Services\Plugins\PluginManager;
use App\Support\SupportPin;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class ClubhosterSeeder extends Seeder
{
    public function run(): void
    {
        $this->seedPlans();
        $this->seedTlds();
        $this->seedAdmin();
        $this->seedPlugins();
        $this->ensurePins();
    }

    private function seedPlans(): void
    {
        if (Plan::query()->count() > 0) {
            return;
        }

        Plan::query()->create([
            'name' => 'Baby Plan',
            'slug' => 'baby',
            'description' => 'For startups and bloggers.',
            'price_monthly' => 499,
            'price_yearly' => 4999,
            'features' => [
                '1 Website',
                '10GB SSD Storage',
                'Free SSL',
                'cPanel',
                '24/7 Support',
            ],
            'is_active' => true,
            'sort_order' => 1,
        ]);

        Plan::query()->create([
            'name' => 'Business Plan',
            'slug' => 'business',
            'description' => 'For growing businesses.',
            'price_monthly' => 999,
            'price_yearly' => 9999,
            'features' => [
                'Unlimited Websites',
                '50GB SSD Storage',
                'Free SSL',
                'Free Migration',
                'Priority Support',
            ],
            'is_active' => true,
            'sort_order' => 2,
        ]);

        Plan::query()->create([
            'name' => 'Cloud Pro',
            'slug' => 'cloud-pro',
            'description' => 'High performance cloud hosting.',
            'price_monthly' => 1999,
            'price_yearly' => 19999,
            'features' => [
                'Unlimited Websites',
                'NVMe Storage',
                'Daily Backups',
                'Free SSL',
                'Premium Support',
            ],
            'is_active' => true,
            'sort_order' => 3,
        ]);
    }

    private function seedTlds(): void
    {
        if (TldPricing::query()->count() > 0) {
            return;
        }

        $defaults = [
            ['tld' => '.com', 'price_yearly' => 3500, 'is_active' => true],
            ['tld' => '.net', 'price_yearly' => 4200, 'is_active' => true],
            ['tld' => '.org', 'price_yearly' => 3900, 'is_active' => true],
            ['tld' => '.pk', 'price_yearly' => 2500, 'is_active' => true],
            ['tld' => '.com.pk', 'price_yearly' => 3000, 'is_active' => true],
        ];

        foreach ($defaults as $row) {
            TldPricing::query()->create($row);
        }
    }

    private function seedAdmin(): void
    {
        $email = env('CLUBHOSTER_ADMIN_EMAIL');
        $password = env('CLUBHOSTER_ADMIN_PASSWORD');

        if (!$email || !$password) {
            // Admin can be created manually later.
            return;
        }

        $admin = User::query()->where('email', $email)->first();
        if ($admin) {
            if (!$admin->is_admin) {
                $admin->update(['is_admin' => true]);
            }
            return;
        }

        User::query()->create([
            'name' => env('CLUBHOSTER_ADMIN_NAME', 'ClubHoster Admin'),
            'email' => $email,
            'password' => Hash::make($password),
            'is_admin' => true,
            'support_pin' => SupportPin::generate((int) config('clubhoster.support_pin_length', 6)),
        ]);
    }

    private function seedPlugins(): void
    {
        // Register bundled plugins into DB. Safe to ignore errors during first install.
        try {
            $manager = app(PluginManager::class);
            $manager->syncDatabase();

            // Enable payment gateways by default (they will only appear on checkout once configured).
            $manager->enable('jazzcash_gateway');
            $manager->enable('paypal_gateway');
            $manager->enable('stripe_gateway');
        } catch (\Throwable $e) {
            // Intentionally silent (installer environments may not yet have plugins path set up).
        }
    }

    private function ensurePins(): void
    {
        User::query()
            ->whereNull('support_pin')
            ->chunk(200, function ($users) {
                foreach ($users as $user) {
                    $user->update([
                        'support_pin' => SupportPin::generate((int) config('clubhoster.support_pin_length', 6)),
                    ]);
                }
            });
    }
}
