<?php

namespace App\Services\Plugins;

use App\Models\Plugin;
use App\Models\PluginSetting;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;

class PluginManager
{
    public function pluginsPath(): string
    {
        return base_path('plugins');
    }

    /**
     * @return array<string,PluginManifest>
     */
    public function discoverManifests(): array
    {
        $path = $this->pluginsPath();
        if (!is_dir($path)) {
            return [];
        }

        $manifests = [];
        foreach (File::directories($path) as $dir) {
            $manifest = PluginManifest::loadFromDirectory($dir);
            if (!$manifest) {
                continue;
            }
            $manifests[$manifest->slug] = $manifest;
        }

        ksort($manifests);
        return $manifests;
    }

    public function syncDatabase(): void
    {
        if (!Schema::hasTable('plugins')) {
            return;
        }

        $manifests = $this->discoverManifests();
        foreach ($manifests as $slug => $manifest) {
            Plugin::query()->updateOrCreate(
                ['slug' => $slug],
                [
                    'name' => $manifest->name,
                    'type' => $manifest->type,
                    'version' => $manifest->version,
                    'author' => $manifest->author,
                    'description' => $manifest->description,
                    'installed_at' => now(),
                ]
            );
        }
    }

    public function getPlugin(string $slug): ?Plugin
    {
        if (!Schema::hasTable('plugins')) {
            return null;
        }

        return Plugin::query()->where('slug', $slug)->first();
    }

    public function enable(string $slug): bool
    {
        $plugin = $this->getPlugin($slug);
        if (!$plugin) {
            return false;
        }

        $plugin->update(['is_enabled' => true]);
        return true;
    }

    public function disable(string $slug): bool
    {
        $plugin = $this->getPlugin($slug);
        if (!$plugin) {
            return false;
        }

        $plugin->update(['is_enabled' => false]);
        return true;
    }

    /**
     * Install a plugin ZIP uploaded from Admin.
     *
     * Expected ZIP structure:
     *  - <PluginFolder>/plugin.json
     *  - <PluginFolder>/bootstrap.php
     */
    public function installZip(UploadedFile $zip): array
    {
        $targetBase = $this->pluginsPath();
        File::ensureDirectoryExists($targetBase);

        $tmpName = 'plugin_' . now()->format('YmdHis') . '_' . Str::random(8) . '.zip';
        $tmpPath = storage_path('app/' . $tmpName);
        $zip->move(dirname($tmpPath), basename($tmpPath));

        $extractTo = storage_path('app/plugin_extract_' . Str::random(10));
        File::ensureDirectoryExists($extractTo);

        $za = new \ZipArchive();
        $open = $za->open($tmpPath);
        if ($open !== true) {
            return ['success' => false, 'message' => 'Could not open ZIP'];
        }

        // Safety: prevent zip-slip
        for ($i = 0; $i < $za->numFiles; $i++) {
            $name = $za->getNameIndex($i);
            if ($name === false) {
                continue;
            }
            if (str_contains($name, '..') || str_starts_with($name, '/') || str_starts_with($name, '\\')) {
                $za->close();
                return ['success' => false, 'message' => 'ZIP contains invalid path entry'];
            }
        }

        $za->extractTo($extractTo);
        $za->close();

        // Detect top-level directory
        $dirs = File::directories($extractTo);
        if (count($dirs) !== 1) {
            File::deleteDirectory($extractTo);
            return ['success' => false, 'message' => 'ZIP must contain exactly one top-level folder'];
        }

        $pluginDir = $dirs[0];
        $manifest = PluginManifest::loadFromDirectory($pluginDir);
        if (!$manifest) {
            File::deleteDirectory($extractTo);
            return ['success' => false, 'message' => 'plugin.json not found or invalid'];
        }

        $slug = $manifest->slug;
        $finalPath = $targetBase . DIRECTORY_SEPARATOR . $slug;

        if (is_dir($finalPath)) {
            File::deleteDirectory($extractTo);
            return ['success' => false, 'message' => 'A plugin with this slug already exists: ' . $slug];
        }

        File::moveDirectory($pluginDir, $finalPath);
        File::deleteDirectory($extractTo);
        File::delete($tmpPath);

        // Sync DB record
        $this->syncDatabase();

        return ['success' => true, 'message' => 'Plugin installed: ' . $slug, 'slug' => $slug];
    }

    public function saveSettings(string $slug, array $values): bool
    {
        $plugin = $this->getPlugin($slug);
        if (!$plugin) {
            return false;
        }

        $manifest = $this->discoverManifests()[$slug] ?? null;
        if (!$manifest) {
            return false;
        }

        DB::transaction(function () use ($plugin, $manifest, $values) {
            foreach ($manifest->configFields as $field) {
                $key = (string) ($field['key'] ?? '');
                if ($key === '') {
                    continue;
                }
                if (!array_key_exists($key, $values)) {
                    continue;
                }

                $encrypted = (bool) ($field['encrypted'] ?? false);
                $val = $values[$key];

                // Do not overwrite existing encrypted secrets with an empty input.
                if ($encrypted && (string) $val === '') {
                    continue;
                }

                $row = PluginSetting::query()->firstOrNew([
                    'plugin_id' => $plugin->id,
                    'key' => $key,
                ]);
                $row->setValue($val, $encrypted);
                $row->save();
            }
        });

        return true;
    }

    public function runPluginMigrations(string $slug): array
    {
        $pluginPath = $this->pluginsPath() . DIRECTORY_SEPARATOR . $slug;
        $migrationsPath = $pluginPath . DIRECTORY_SEPARATOR . 'database' . DIRECTORY_SEPARATOR . 'migrations';

        if (!is_dir($migrationsPath)) {
            return ['success' => false, 'message' => 'No migrations found for plugin'];
        }

        // Run plugin migrations by path (works without service provider)
        Artisan::call('migrate', [
            '--path' => 'plugins/' . $slug . '/database/migrations',
            '--force' => true,
        ]);

        return ['success' => true, 'message' => 'Plugin migrations executed.'];
    }
}
