<?php

namespace App\Services\Payments;

use App\Models\Invoice;
use App\Models\Transaction;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class JazzCashService
{
    /**
     * Perform a DoMWalletTransaction request.
     *
     * This method:
     *  - builds payload
     *  - computes pp_SecureHash
     *  - calls JazzCash endpoint
     *
     * It DOES NOT change invoice/order/service status and DOES NOT create a transaction row.
     */
    public function doMWalletTransaction(Invoice $invoice, array $payer, array $override = []): array
    {
        $merchantId = trim((string) ($override['merchant_id'] ?? config('jazzcash.merchant_id')));
        $password = trim((string) ($override['password'] ?? config('jazzcash.password')));
        $salt = trim((string) ($override['integrity_salt'] ?? config('jazzcash.integrity_salt')));

        if ($merchantId === '' || $password === '' || $salt === '') {
            return [
                'success' => false,
                'message' => 'JazzCash credentials are not configured.',
            ];
        }

        $currency = $invoice->currency ?: (string) ($override['currency'] ?? config('clubhoster.currency', 'PKR'));

        // JazzCash expects amount in "paisa" (multiply by 100)
        $amountPaisa = (int) round(((float) $invoice->total) * 100);

        $mobile = $this->normalizeMobile((string) Arr::get($payer, 'mobile', ''));
        $cnic = $this->normalizeCnic((string) Arr::get($payer, 'cnic', ''));

        if ($mobile === '' || $cnic === '') {
            return [
                'success' => false,
                'message' => 'JazzCash Mobile Number and CNIC are required.',
            ];
        }

        $txnRef = $override['txn_ref'] ?? $this->generateTxnRefNo();

        $expiryMinutes = (int) ($override['expiry_minutes'] ?? config('jazzcash.expiry_minutes', 1440));
        $expiryMinutes = max(5, min(7 * 24 * 60, $expiryMinutes));

        $payload = [
            'pp_Amount' => $amountPaisa,
            'pp_BillReference' => $invoice->number,
            'pp_CNIC' => $cnic,
            'pp_Description' => 'ClubHoster Invoice ' . $invoice->number,
            'pp_Language' => 'EN',
            'pp_MerchantID' => $merchantId,
            'pp_MobileNumber' => $mobile,
            'pp_Password' => $password,
            'pp_TxnCurrency' => $currency,
            'pp_TxnDateTime' => now()->format('YmdHis'),
            'pp_TxnExpiryDateTime' => now()->addMinutes($expiryMinutes)->format('YmdHis'),
            'pp_TxnRefNo' => $txnRef,
            'ppmpf_1' => (string) $invoice->id,
            'ppmpf_2' => (string) $invoice->user_id,
            'ppmpf_3' => 'clubhoster',
            'ppmpf_4' => '',
            'ppmpf_5' => '',
        ];

        // Compute secure hash as used in many REST v2.0 implementations:
        //   ksort(fields); hashString = salt . '&' . implode('&', array_values(fields));
        //   pp_SecureHash = hash_hmac('sha256', hashString, salt)
        ksort($payload);
        $hashString = $salt . '&' . implode('&', array_values($payload));
        $payload['pp_SecureHash'] = hash_hmac('sha256', $hashString, $salt);

        $url = $this->endpointUrl($override);

        try {
            $response = Http::timeout(45)
                ->withHeaders(['Content-Type' => 'application/json'])
                ->post($url, $payload);

            $respJson = $response->json() ?: [];

            $code = (string) ($respJson['pp_ResponseCode'] ?? $respJson['responseCode'] ?? '');
            $message = (string) ($respJson['pp_ResponseMessage'] ?? $respJson['responseMessage'] ?? '');

            return [
                'success' => $code === '000',
                'txn_ref' => $txnRef,
                'payload' => $payload,
                'response' => $respJson,
                'response_code' => $code,
                'message' => $message !== '' ? $message : ($code === '000' ? 'Payment successful.' : 'Payment failed.'),
            ];
        } catch (\Throwable $e) {
            return [
                'success' => false,
                'txn_ref' => $txnRef,
                'payload' => $payload,
                'message' => 'Payment error: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Legacy helper retained for backward compatibility.
     * Creates a transaction row + marks invoice paid if successful.
     */
    public function payInvoiceViaMWallet(Invoice $invoice, array $payer): array
    {
        if ($invoice->isPaid()) {
            return [
                'success' => true,
                'message' => 'Invoice already paid.',
            ];
        }

        $result = $this->doMWalletTransaction($invoice, $payer);

        $currency = $invoice->currency ?: config('clubhoster.currency', 'PKR');
        $txnRef = (string) ($result['txn_ref'] ?? $this->generateTxnRefNo());

        $transaction = Transaction::query()->create([
            'invoice_id' => $invoice->id,
            'user_id' => $invoice->user_id,
            'gateway' => 'jazzcash_mwallet',
            'txn_ref' => $txnRef,
            'amount' => (float) $invoice->total,
            'currency' => $currency,
            'status' => ($result['success'] ?? false) ? 'success' : (($result['response_code'] ?? null) ? 'failed' : 'error'),
            'response_code' => $result['response_code'] ?? null,
            'response_message' => $result['message'] ?? null,
            'raw_request' => $result['payload'] ?? null,
            'raw_response' => $result['response'] ?? null,
        ]);

        if (($result['success'] ?? false) === true) {
            $invoice->update([
                'status' => 'paid',
                'paid_at' => now(),
            ]);

            if ($invoice->order) {
                $invoice->order->update(['status' => 'paid']);

                $invoice->user->services()
                    ->where('order_id', $invoice->order_id)
                    ->where('status', 'pending')
                    ->update(['status' => 'active']);
            }
        }

        return [
            'success' => (bool) ($result['success'] ?? false),
            'message' => (string) ($result['message'] ?? (($result['success'] ?? false) ? 'Payment successful.' : 'Payment failed.')),
            'transaction_id' => $transaction->id,
            'response' => $result['response'] ?? null,
        ];
    }

    private function endpointUrl(array $override = []): string
    {
        $env = (string) ($override['environment'] ?? config('jazzcash.environment', 'sandbox'));
        $base = (string) ($override['base_url'] ?? config('jazzcash.base_url'));

        if (!$base) {
            $base = $env === 'live'
                ? 'https://payments.jazzcash.com.pk/ApplicationAPI/API'
                : 'https://sandbox.jazzcash.com.pk/ApplicationAPI/API';
        }

        $base = rtrim($base, '/');
        $version = (string) ($override['api_version'] ?? config('jazzcash.api_version', '2.0'));

        return $base . '/' . $version . '/Purchase/DoMWalletTransaction';
    }

    private function generateTxnRefNo(): string
    {
        // JazzCash examples commonly use: T + yyyymmddhhMMss + random
        return 'T' . now()->format('YmdHis') . random_int(100, 999);
    }

    private function normalizeMobile(string $mobile): string
    {
        $mobile = preg_replace('/\D+/', '', $mobile) ?? '';

        // Allow both 03xxxxxxxxx and 92xxxxxxxxxx formats
        if (Str::startsWith($mobile, '0') && strlen($mobile) === 11) {
            return $mobile;
        }
        if (Str::startsWith($mobile, '92') && strlen($mobile) === 12) {
            return $mobile;
        }

        // Fallback: return digits as-is
        return $mobile;
    }

    private function normalizeCnic(string $cnic): string
    {
        $cnic = preg_replace('/\D+/', '', $cnic) ?? '';
        if ($cnic === '') {
            return '';
        }

        // Many MWallet flows use last 6 digits.
        if (strlen($cnic) > 6) {
            $cnic = substr($cnic, -6);
        }

        return $cnic;
    }
}
