<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Services\Payments\GatewayRegistry;
use Illuminate\Http\Request;

class PaymentsController extends Controller
{
    public function handleReturn(Request $request, string $gateway)
    {
        $gw = GatewayRegistry::instance()->get($gateway);
        if (!$gw) {
            abort(404);
        }

        $result = $gw->handleReturn($request);

        $success = (bool) ($result['success'] ?? false);
        $message = (string) ($result['message'] ?? ($success ? 'Payment successful.' : 'Payment failed.'));

        $invoiceId = $request->query('invoice') ?? ($result['invoice_id'] ?? null);
        $invoice = $invoiceId ? Invoice::query()->find($invoiceId) : null;

        if ($success) {
            if ($request->user()) {
                if ($invoice) {
                    return redirect()->route('client.invoices.show', $invoice)->with('success', $message);
                }
                return redirect()->route('client.dashboard')->with('success', $message);
            }

            // Guest return: send to login with a notice.
            return redirect()->route('login')->with('success', $message);
        }

        // Failure
        if ($invoice) {
            return redirect()->route('client.invoices.show', $invoice)->with('error', $message);
        }

        return redirect()->route('home')->with('error', $message);
    }

    public function cancel(Request $request, string $gateway)
    {
        $invoiceId = $request->query('invoice');
        $invoice = $invoiceId ? Invoice::query()->find($invoiceId) : null;

        if ($invoice && $request->user()) {
            return redirect()->route('client.invoices.show', $invoice)->with('error', 'Payment cancelled.');
        }

        return redirect()->route('home')->with('error', 'Payment cancelled.');
    }

    public function webhook(Request $request, string $gateway)
    {
        $gw = GatewayRegistry::instance()->get($gateway);
        if (!$gw) {
            return response()->json(['success' => false, 'message' => 'Unknown gateway'], 404);
        }

        $result = $gw->handleWebhook($request);

        $success = (bool) ($result['success'] ?? false);
        $status = $success ? 200 : 400;

        return response()->json($result, $status);
    }
}
