<?php

namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Services\Payments\GatewayRegistry;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class InvoicesController extends Controller
{
    public function index(Request $request)
    {
        $invoices = $request->user()->invoices()->latest()->paginate(15);
        return view('client.invoices.index', compact('invoices'));
    }

    public function show(Request $request, Invoice $invoice)
    {
        $this->authorizeInvoice($request, $invoice);
        $invoice->load(['items', 'transactions' => function ($q) { $q->latest(); }]);

        $gateways = array_values(array_filter(
            GatewayRegistry::instance()->all(),
            fn ($gw) => $gw->isConfigured()
        ));

        return view('client.invoices.show', compact('invoice', 'gateways'));
    }

    public function pay(Request $request, Invoice $invoice)
    {
        $this->authorizeInvoice($request, $invoice);

        if ($invoice->isPaid()) {
            return redirect()->route('client.invoices.show', $invoice)->with('success', 'Invoice is already paid.');
        }

        $availableGateways = GatewayRegistry::instance()->all();
        $gatewaySlugs = array_keys($availableGateways);

        $selectedGatewaySlug = (string) $request->input('payment_gateway', '');
        $gateway = $availableGateways[$selectedGatewaySlug] ?? null;

        $rules = [
            'payment_gateway' => ['required', 'string', Rule::in($gatewaySlugs)],
        ];

        if ($gateway) {
            $rules = array_merge($rules, $gateway->validationRules());
        }

        $data = $request->validate($rules);

        $gateway = GatewayRegistry::instance()->get((string) $data['payment_gateway']);
        if (!$gateway || !$gateway->isConfigured()) {
            return back()->with('error', 'Selected payment gateway is not available or not configured.');
        }

        // Ensure invoice has a payment token for redirect-based gateways
        $meta = $invoice->meta ?? [];
        if (empty($meta['payment_token'])) {
            $meta['payment_token'] = Str::random(32);
            $invoice->meta = $meta;
            $invoice->save();
        }

        $payResult = $gateway->initiate($invoice, $data);

        if (($payResult['success'] ?? false) === true) {
            if (!empty($payResult['redirect_url'])) {
                return redirect()->away((string) $payResult['redirect_url']);
            }
            return redirect()->route('client.invoices.show', $invoice)->with('success', $payResult['message'] ?? 'Payment received successfully.');
        }

        return redirect()->route('client.invoices.show', $invoice)
            ->with('error', $payResult['message'] ?? 'Payment failed');
    }

    private function authorizeInvoice(Request $request, Invoice $invoice): void
    {
        $user = $request->user();
        if (!$user) {
            abort(401);
        }
        if ($user->is_admin) {
            return;
        }
        if ($invoice->user_id !== $user->id) {
            abort(403);
        }
    }
}
