<?php

namespace App\Http\Controllers;

use App\Models\Domain;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Plan;
use App\Models\Service;
use App\Models\TldPricing;
use App\Models\User;
use App\Services\Payments\GatewayRegistry;
use App\Support\SupportPin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class CheckoutController extends Controller
{
    public function show(Request $request)
    {
        $plans = Plan::query()->where('is_active', true)->orderBy('sort_order')->get();
        $tlds = TldPricing::query()->where('is_active', true)->orderBy('tld')->get();

        $selectedPlanId = (int) $request->query('plan', 0);
        $domain = trim((string) $request->query('domain', ''));

        // Persist selection: user enters once, it stays selected.
        if ($selectedPlanId > 0) {
            $request->session()->put('checkout.plan', $selectedPlanId);
        }
        if ($domain !== '') {
            $request->session()->put('checkout.domain', $domain);
        }

        $selectedPlanId = $selectedPlanId > 0 ? $selectedPlanId : (int) $request->session()->get('checkout.plan', 0);
        $domain = $domain !== '' ? $domain : (string) $request->session()->get('checkout.domain', '');

        // Only show configured gateways on checkout
        $gateways = array_values(array_filter(
            GatewayRegistry::instance()->all(),
            fn ($gw) => $gw->isConfigured()
        ));

        return view('checkout.index', [
            'plans' => $plans,
            'tlds' => $tlds,
            'selectedPlanId' => $selectedPlanId,
            'domain' => $domain,
            'gateways' => $gateways,
        ]);
    }

    public function submit(Request $request)
    {
        $user = $request->user();

        $availableGateways = GatewayRegistry::instance()->all();
        $gatewaySlugs = array_keys($availableGateways);

        $selectedGatewaySlug = (string) $request->input('payment_gateway', '');
        $gateway = $availableGateways[$selectedGatewaySlug] ?? null;

        $rules = [
            'plan_id' => ['required', 'integer', 'exists:plans,id'],
            'billing_cycle' => ['required', 'in:monthly,yearly'],
            'domain_option' => ['required', 'in:register,transfer,use_own,none'],
            'domain_name' => ['nullable', 'string', 'max:255'],
            'domain_years' => ['nullable', 'integer', 'min:1', 'max:10'],

            'payment_gateway' => ['required', 'string', Rule::in($gatewaySlugs)],
        ];

        if ($gateway) {
            $rules = array_merge($rules, $gateway->validationRules());
        }

        if (!$user) {
            $rules = array_merge($rules, [
                'name' => ['required', 'string', 'max:120'],
                'email' => ['required', 'email', 'max:190', 'unique:users,email'],
                'password' => ['required', 'string', 'min:8', 'confirmed'],
                'phone' => ['nullable', 'string', 'max:40'],
            ]);
        }

        $data = $request->validate($rules);

        $selectedGatewaySlug = (string) $data['payment_gateway'];
        $gateway = GatewayRegistry::instance()->get($selectedGatewaySlug);
        if (!$gateway || !$gateway->isConfigured()) {
            return back()->withErrors(['payment_gateway' => 'Selected payment gateway is not available or not configured.'])->withInput();
        }

        // Basic domain normalization
        $domainName = trim((string) ($data['domain_name'] ?? ''));
        $domainOption = $data['domain_option'];
        if ($domainOption !== 'none') {
            if ($domainName === '') {
                return back()->withErrors(['domain_name' => 'Domain is required for the selected option.'])->withInput();
            }
            $domainName = $this->normalizeDomain($domainName);
        } else {
            $domainName = '';
        }

        $plan = Plan::query()->findOrFail((int) $data['plan_id']);
        $billingCycle = $data['billing_cycle'];

        $domainYears = (int) ($data['domain_years'] ?? 1);
        $domainYears = max(1, min(10, $domainYears));

        $currency = config('clubhoster.currency', 'PKR');
        $taxPercent = (float) config('clubhoster.tax_percent', 0);

        $planPrice = $this->calculatePlanPrice($plan, $billingCycle);
        $domainPrice = 0.00;

        if ($domainOption === 'register') {
            $domainPrice = $this->calculateDomainPrice($domainName) * $domainYears;
        }

        $subtotal = (float) ($planPrice + $domainPrice);
        $tax = round($subtotal * ($taxPercent / 100), 2);
        $total = round($subtotal + $tax, 2);

        /** @var \App\Models\Invoice|null $invoice */
        $invoice = null;

        DB::transaction(function () use (&$invoice, $data, &$user, $plan, $billingCycle, $domainOption, $domainName, $domainYears, $planPrice, $domainPrice, $subtotal, $tax, $total, $currency, $taxPercent, $selectedGatewaySlug) {
            // Create account if needed
            if (!$user) {
                $user = User::query()->create([
                    'name' => $data['name'],
                    'email' => $data['email'],
                    'password' => Hash::make($data['password']),
                    'phone' => $data['phone'] ?? null,
                    'status' => 'active',
                    'support_pin' => SupportPin::generate((int) config('clubhoster.support_pin_length', 6)),
                    'is_admin' => false,
                ]);

                Auth::login($user);
            }

            // Create order
            $order = Order::query()->create([
                'user_id' => $user->id,
                'status' => 'pending',
                'subtotal' => $subtotal,
                'tax' => $tax,
                'total' => $total,
                'currency' => $currency,
            ]);

            // Optional domain record
            $domainModel = null;
            if ($domainOption !== 'none' && $domainName !== '') {
                $domainModel = Domain::query()->create([
                    'user_id' => $user->id,
                    'domain' => $domainName,
                    'years' => $domainYears,
                    'status' => $domainOption === 'register' ? 'pending_registration' : ($domainOption === 'transfer' ? 'pending_transfer' : 'linked'),
                    'registrar' => null,
                    'expires_at' => null,
                    'meta' => [
                        'option' => $domainOption,
                    ],
                ]);

                if ($domainOption === 'register') {
                    OrderItem::query()->create([
                        'order_id' => $order->id,
                        'item_type' => 'domain',
                        'reference_id' => $domainModel->id,
                        'name' => $domainName . ' (' . $domainYears . ' year' . ($domainYears > 1 ? 's' : '') . ')',
                        'qty' => 1,
                        'unit_price' => $domainPrice,
                        'total' => $domainPrice,
                        'meta' => [
                            'years' => $domainYears,
                        ],
                    ]);
                }
            }

            // Hosting order item
            OrderItem::query()->create([
                'order_id' => $order->id,
                'item_type' => 'hosting',
                'reference_id' => $plan->id,
                'name' => $plan->name . ' (' . ucfirst($billingCycle) . ')',
                'qty' => 1,
                'unit_price' => $planPrice,
                'total' => $planPrice,
                'meta' => [
                    'billing_cycle' => $billingCycle,
                ],
            ]);

            // Provision a pending service record
            $nextDue = now()->addMonth();
            if ($billingCycle === 'yearly') {
                $nextDue = now()->addYear();
            }

            Service::query()->create([
                'user_id' => $user->id,
                'order_id' => $order->id,
                'type' => 'hosting',
                'plan_id' => $plan->id,
                'domain_id' => $domainModel?->id,
                'billing_cycle' => $billingCycle,
                'status' => 'pending',
                'next_due_date' => $nextDue,
                'meta' => [
                    'domain' => $domainName,
                    'domain_option' => $domainOption,
                ],
            ]);

            // Create invoice
            $invoice = Invoice::query()->create([
                'user_id' => $user->id,
                'order_id' => $order->id,
                'number' => $this->generateInvoiceNumber(),
                'status' => 'unpaid',
                'due_date' => now()->addDays(1),
                'subtotal' => $subtotal,
                'tax' => $tax,
                'total' => $total,
                'currency' => $currency,
                'meta' => [
                    'payment_gateway' => $selectedGatewaySlug,
                    'payment_token' => Str::random(32),
                ],
            ]);

            // Invoice items
            InvoiceItem::query()->create([
                'invoice_id' => $invoice->id,
                'description' => $plan->name . ' Hosting (' . ucfirst($billingCycle) . ')',
                'qty' => 1,
                'unit_price' => $planPrice,
                'total' => $planPrice,
                'meta' => [
                    'plan_id' => $plan->id,
                    'billing_cycle' => $billingCycle,
                ],
            ]);

            if ($domainOption === 'register' && $domainPrice > 0) {
                InvoiceItem::query()->create([
                    'invoice_id' => $invoice->id,
                    'description' => 'Domain Registration: ' . $domainName . ' (' . $domainYears . ' year' . ($domainYears > 1 ? 's' : '') . ')',
                    'qty' => 1,
                    'unit_price' => $domainPrice,
                    'total' => $domainPrice,
                    'meta' => [
                        'domain_id' => $domainModel?->id,
                        'years' => $domainYears,
                    ],
                ]);
            }

            if ($tax > 0) {
                InvoiceItem::query()->create([
                    'invoice_id' => $invoice->id,
                    'description' => 'Tax',
                    'qty' => 1,
                    'unit_price' => $tax,
                    'total' => $tax,
                    'meta' => [
                        'percent' => $taxPercent,
                    ],
                ]);
            }
        });

        // Initiate payment
        $payment = $gateway->initiate($invoice, $data);

        if (($payment['success'] ?? false) === true) {
            $redirect = $payment['redirect_url'] ?? null;
            if ($redirect) {
                return redirect()->away($redirect);
            }
            return redirect()->route('client.dashboard')->with('success', $payment['message'] ?? 'Payment successful.');
        }

        // Failed payment - keep invoice UNPAID and allow retry from invoice page
        return redirect()->route('client.invoices.show', $invoice)->withErrors([
            'payment' => $payment['message'] ?? 'Payment failed. Please try again.',
        ]);
    }

    private function normalizeDomain(string $domain): string
    {
        $domain = strtolower(trim($domain));
        $domain = preg_replace('/^https?:\/\//', '', $domain) ?? $domain;
        $domain = preg_replace('/\/.*/', '', $domain) ?? $domain;
        $domain = preg_replace('/\s+/', '', $domain) ?? $domain;
        return $domain;
    }

    private function calculatePlanPrice(Plan $plan, string $billingCycle): float
    {
        return $billingCycle === 'yearly'
            ? (float) $plan->price_yearly
            : (float) $plan->price_monthly;
    }

    private function calculateDomainPrice(string $domain): float
    {
        // Extract TLD from domain
        $dotPos = strrpos($domain, '.');
        $tld = $dotPos === false ? '' : substr($domain, $dotPos);

        if ($tld === '') {
            return (float) config('clubhoster.domain.default_price_yearly', 0);
        }

        $pricing = TldPricing::query()->where('tld', $tld)->where('is_active', true)->first();
        return $pricing ? (float) $pricing->price_yearly : (float) config('clubhoster.domain.default_price_yearly', 0);
    }

    private function generateInvoiceNumber(): string
    {
        // Format: CH-YYYYMMDD-XXXX
        $date = now()->format('Ymd');
        $rand = random_int(1000, 9999);
        return 'CH-' . $date . '-' . $rand;
    }
}
