<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Plugin;
use App\Services\Plugins\PluginManager;
use Illuminate\Http\Request;

class PluginsController extends Controller
{
    public function index(Request $request)
    {
        $this->ensureAdmin($request);

        $manager = app(PluginManager::class);
        $manager->syncDatabase();

        $manifests = $manager->discoverManifests();
        $records = Plugin::query()->with('settings')->whereIn('slug', array_keys($manifests))->get()->keyBy('slug');

        $plugins = [];
        foreach ($manifests as $slug => $manifest) {
            $plugins[] = [
                'slug' => $slug,
                'manifest' => $manifest,
                'record' => $records[$slug] ?? null,
            ];
        }

        return view('admin.plugins.index', [
            'plugins' => $plugins,
        ]);
    }

    public function upload(Request $request)
    {
        $this->ensureAdmin($request);

        $request->validate([
            'zip' => ['required', 'file', 'mimes:zip'],
        ]);

        $manager = app(PluginManager::class);
        $result = $manager->installZip($request->file('zip'));

        return back()->with($result['success'] ? 'success' : 'error', $result['message']);
    }

    public function enable(Request $request, string $slug)
    {
        $this->ensureAdmin($request);

        $ok = app(PluginManager::class)->enable($slug);
        return back()->with($ok ? 'success' : 'error', $ok ? 'Plugin enabled.' : 'Plugin not found.');
    }

    public function disable(Request $request, string $slug)
    {
        $this->ensureAdmin($request);

        $ok = app(PluginManager::class)->disable($slug);
        return back()->with($ok ? 'success' : 'error', $ok ? 'Plugin disabled.' : 'Plugin not found.');
    }

    public function migrate(Request $request, string $slug)
    {
        $this->ensureAdmin($request);

        $result = app(PluginManager::class)->runPluginMigrations($slug);
        return back()->with($result['success'] ? 'success' : 'error', $result['message']);
    }

    public function saveSettings(Request $request, string $slug)
    {
        $this->ensureAdmin($request);

        $ok = app(PluginManager::class)->saveSettings($slug, $request->all());
        return back()->with($ok ? 'success' : 'error', $ok ? 'Settings saved.' : 'Could not save settings.');
    }

    private function ensureAdmin(Request $request): void
    {
        $user = $request->user();
        if (!$user || !$user->is_admin) {
            abort(403);
        }
    }
}
