<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Plan;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PlansController extends Controller
{
    public function index(Request $request)
    {
        $this->ensureAdmin($request);

        $plans = Plan::query()->orderBy('sort_order')->paginate(20);
        return view('admin.plans.index', compact('plans'));
    }

    public function create(Request $request)
    {
        $this->ensureAdmin($request);
        return view('admin.plans.create');
    }

    public function store(Request $request)
    {
        $this->ensureAdmin($request);

        $data = $request->validate([
            'name' => ['required', 'string', 'max:120'],
            'slug' => ['nullable', 'string', 'max:120'],
            'description' => ['nullable', 'string'],
            'price_monthly' => ['required', 'numeric', 'min:0'],
            'price_yearly' => ['required', 'numeric', 'min:0'],
            'features' => ['nullable', 'string'], // newline separated
            'is_active' => ['nullable', 'boolean'],
            'sort_order' => ['nullable', 'integer', 'min:0'],
        ]);

        $features = [];
        if (!empty($data['features'])) {
            $features = array_values(array_filter(array_map('trim', preg_split('/\r\n|\r|\n/', $data['features']))));
        }

        $slug = trim((string) ($data['slug'] ?? ''));
        if ($slug === '') {
            $slug = Str::slug($data['name']);
        }

        Plan::query()->create([
            'name' => $data['name'],
            'slug' => $slug,
            'description' => $data['description'] ?? null,
            'price_monthly' => (float) $data['price_monthly'],
            'price_yearly' => (float) $data['price_yearly'],
            'features' => $features,
            'is_active' => (bool) ($data['is_active'] ?? true),
            'sort_order' => (int) ($data['sort_order'] ?? 0),
        ]);

        return redirect()->route('admin.plans.index')->with('success', 'Plan created.');
    }

    public function edit(Request $request, Plan $plan)
    {
        $this->ensureAdmin($request);
        return view('admin.plans.edit', compact('plan'));
    }

    public function update(Request $request, Plan $plan)
    {
        $this->ensureAdmin($request);

        $data = $request->validate([
            'name' => ['required', 'string', 'max:120'],
            'slug' => ['required', 'string', 'max:120'],
            'description' => ['nullable', 'string'],
            'price_monthly' => ['required', 'numeric', 'min:0'],
            'price_yearly' => ['required', 'numeric', 'min:0'],
            'features' => ['nullable', 'string'],
            'is_active' => ['nullable', 'boolean'],
            'sort_order' => ['nullable', 'integer', 'min:0'],
        ]);

        $features = [];
        if (!empty($data['features'])) {
            $features = array_values(array_filter(array_map('trim', preg_split('/\r\n|\r|\n/', $data['features']))));
        }

        $plan->update([
            'name' => $data['name'],
            'slug' => $data['slug'],
            'description' => $data['description'] ?? null,
            'price_monthly' => (float) $data['price_monthly'],
            'price_yearly' => (float) $data['price_yearly'],
            'features' => $features,
            'is_active' => (bool) ($data['is_active'] ?? false),
            'sort_order' => (int) ($data['sort_order'] ?? 0),
        ]);

        return redirect()->route('admin.plans.index')->with('success', 'Plan updated.');
    }

    public function destroy(Request $request, Plan $plan)
    {
        $this->ensureAdmin($request);
        $plan->delete();
        return redirect()->route('admin.plans.index')->with('success', 'Plan deleted.');
    }

    private function ensureAdmin(Request $request): void
    {
        $user = $request->user();
        if (!$user || !$user->is_admin) {
            abort(403);
        }
    }
}
