<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use Illuminate\Http\Request;

class InvoicesController extends Controller
{
    public function index(Request $request)
    {
        $this->ensureAdmin($request);

        $q = trim((string) $request->query('q'));

        $invoices = Invoice::query()
            ->when($q !== '', function ($query) use ($q) {
                $query->where('number', 'like', "%{$q}%")
                    ->orWhere('status', 'like', "%{$q}%");
            })
            ->orderByDesc('id')
            ->paginate(20)
            ->withQueryString();

        return view('admin.invoices.index', compact('invoices', 'q'));
    }

    public function show(Request $request, Invoice $invoice)
    {
        $this->ensureAdmin($request);

        $invoice->load(['user', 'items', 'transactions' => function ($q) { $q->latest(); }]);

        return view('admin.invoices.show', compact('invoice'));
    }

    public function markPaid(Request $request, Invoice $invoice)
    {
        $this->ensureAdmin($request);

        $invoice->update([
            'status' => 'paid',
            'paid_at' => now(),
        ]);

        return redirect()->route('admin.invoices.show', $invoice)->with('success', 'Invoice marked as paid.');
    }

    public function markUnpaid(Request $request, Invoice $invoice)
    {
        $this->ensureAdmin($request);

        $invoice->update([
            'status' => 'unpaid',
            'paid_at' => null,
        ]);

        return redirect()->route('admin.invoices.show', $invoice)->with('success', 'Invoice marked as unpaid.');
    }

    private function ensureAdmin(Request $request): void
    {
        $user = $request->user();
        if (!$user || !$user->is_admin) {
            abort(403);
        }
    }
}
