<?php

namespace App\Console\Commands;

use App\Models\Domain;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Transaction;
use App\Models\User;
use App\Support\SupportPin;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;

class ImportClientExec extends Command
{
    protected $signature = 'clubhoster:import-clientexec
        {--dry-run : Do not write anything, only show what would happen}
        {--only-users : Import users only}
        {--only-billing : Import invoices/transactions only (requires users mapped)}
        {--chunk=500 : Chunk size for import}
    ';

    protected $description = 'Import core ClientExec data (users, invoices, transactions, domains) into ClubHoster.';

    public function handle(): int
    {
        $dryRun = (bool) $this->option('dry-run');
        $onlyUsers = (bool) $this->option('only-users');
        $onlyBilling = (bool) $this->option('only-billing');
        $chunk = (int) $this->option('chunk');
        $chunk = max(50, min(5000, $chunk));

        $this->info('ClientExec import started' . ($dryRun ? ' (DRY RUN)' : ''));

        $ce = $this->clientExecConnection();

        if (!Schema::connection($ce)->hasTable('users')) {
            $this->error('ClientExec table "users" not found. Check CLIENTEXEC_DB_* env vars.');
            return self::FAILURE;
        }

        if (!$onlyBilling) {
            $this->importUsers($ce, $dryRun, $chunk);
        }

        if (!$onlyUsers) {
            $this->importDomains($ce, $dryRun, $chunk);
            $this->importInvoices($ce, $dryRun, $chunk);
            $this->importTransactions($ce, $dryRun, $chunk);
        }

        $this->info('ClientExec import completed.');
        return self::SUCCESS;
    }

    /**
     * Create a runtime DB connection called "clientexec" from env vars.
     * This avoids requiring manual edits to config/database.php.
     */
    private function clientExecConnection(): string
    {
        $name = 'clientexec';

        config([
            "database.connections.$name" => [
                'driver' => 'mysql',
                'host' => env('CLIENTEXEC_DB_HOST', '127.0.0.1'),
                'port' => env('CLIENTEXEC_DB_PORT', '3306'),
                'database' => env('CLIENTEXEC_DB_DATABASE', ''),
                'username' => env('CLIENTEXEC_DB_USERNAME', ''),
                'password' => env('CLIENTEXEC_DB_PASSWORD', ''),
                'unix_socket' => env('CLIENTEXEC_DB_SOCKET', ''),
                'charset' => 'utf8mb4',
                'collation' => 'utf8mb4_unicode_ci',
                'prefix' => env('CLIENTEXEC_DB_PREFIX', ''),
                'strict' => false,
                'engine' => null,
            ],
        ]);

        // Validate connection early
        DB::connection($name)->getPdo();

        return $name;
    }

    private function importUsers(string $ce, bool $dryRun, int $chunk): void
    {
        $this->line('--- Importing users ---');

        $cols = Schema::connection($ce)->getColumnListing('users');
        $colId = $this->pickColumn($cols, ['id', 'userid', 'user_id']);
        $colEmail = $this->pickColumn($cols, ['email', 'emailaddress', 'email_address']);
        $colFirst = $this->pickColumn($cols, ['firstname', 'first_name', 'fname']);
        $colLast = $this->pickColumn($cols, ['lastname', 'last_name', 'lname']);
        $colName = $this->pickColumn($cols, ['name', 'fullname']);
        $colPhone = $this->pickColumn($cols, ['phonenumber', 'phone', 'cellphone', 'mobile']);
        $colCompany = $this->pickColumn($cols, ['companyname', 'company']);
        $colGroup = $this->pickColumn($cols, ['groupid', 'group_id']);

        if (!$colId || !$colEmail) {
            $this->error('Could not detect required columns in ClientExec users table (id/email).');
            return;
        }

        $query = DB::connection($ce)->table('users');

        // Prefer importing clients only (ClientExec usually uses groupid > 1 for clients)
        if ($colGroup) {
            $query->where($colGroup, '>', 1);
        }

        $total = (clone $query)->count();
        $this->info("Found $total ClientExec client records.");

        $bar = $this->output->createProgressBar($total);
        $bar->start();

        $query->orderBy($colId)->chunk($chunk, function ($rows) use ($dryRun, $bar, $colId, $colEmail, $colFirst, $colLast, $colName, $colPhone, $colCompany) {
            foreach ($rows as $row) {
                $legacyId = (string) $row->{$colId};
                $email = strtolower(trim((string) $row->{$colEmail}));
                if ($email === '') {
                    $bar->advance();
                    continue;
                }

                $name = '';
                if ($colName && !empty($row->{$colName})) {
                    $name = trim((string) $row->{$colName});
                }
                if ($name === '') {
                    $first = $colFirst ? trim((string) ($row->{$colFirst} ?? '')) : '';
                    $last = $colLast ? trim((string) ($row->{$colLast} ?? '')) : '';
                    $name = trim($first . ' ' . $last);
                }
                if ($name === '' && $colCompany && !empty($row->{$colCompany})) {
                    $name = trim((string) $row->{$colCompany});
                }
                if ($name === '') {
                    $name = $email;
                }

                $phone = $colPhone ? trim((string) ($row->{$colPhone} ?? '')) : '';
                $company = $colCompany ? trim((string) ($row->{$colCompany} ?? '')) : '';

                $existing = User::query()->where('email', $email)->first();
                if ($existing) {
                    if (!$dryRun) {
                        $existing->update([
                            'legacy_clientexec_id' => $existing->legacy_clientexec_id ?: $legacyId,
                            'phone' => $existing->phone ?: ($phone ?: null),
                            'company' => $existing->company ?: ($company ?: null),
                            'support_pin' => $existing->support_pin ?: SupportPin::generate((int) config('clubhoster.support_pin_length', 6)),
                        ]);
                    }

                    $bar->advance();
                    continue;
                }

                if (!$dryRun) {
                    User::query()->create([
                        'name' => $name,
                        'email' => $email,
                        // We cannot reliably convert ClientExec password hashes without knowing hashing settings.
                        // Set a random password and let the user reset via the new system.
                        'password' => Hash::make(Str::random(32)),
                        'is_admin' => false,
                        'support_pin' => SupportPin::generate((int) config('clubhoster.support_pin_length', 6)),
                        'phone' => $phone ?: null,
                        'company' => $company ?: null,
                        'legacy_clientexec_id' => $legacyId,
                    ]);
                }

                $bar->advance();
            }
        });

        $bar->finish();
        $this->newLine(2);
        $this->info('Users import done.');
    }

    private function importDomains(string $ce, bool $dryRun, int $chunk): void
    {
        if (!Schema::connection($ce)->hasTable('domains')) {
            $this->line('--- Skipping domains (ClientExec table "domains" not found) ---');
            return;
        }

        $this->line('--- Importing domains ---');

        $cols = Schema::connection($ce)->getColumnListing('domains');
        $colId = $this->pickColumn($cols, ['id', 'domainid']);
        $colUser = $this->pickColumn($cols, ['userid', 'user_id', 'customerid', 'clientid']);
        $colDomain = $this->pickColumn($cols, ['domainname', 'domain', 'name']);
        $colStatus = $this->pickColumn($cols, ['status']);

        if (!$colId || !$colUser || !$colDomain) {
            $this->warn('Could not detect required columns in ClientExec domains table.');
            return;
        }

        $query = DB::connection($ce)->table('domains');
        $total = (clone $query)->count();
        $this->info("Found $total ClientExec domain records.");

        $bar = $this->output->createProgressBar($total);
        $bar->start();

        $query->orderBy($colId)->chunk($chunk, function ($rows) use ($dryRun, $bar, $colUser, $colDomain, $colStatus) {
            foreach ($rows as $row) {
                $legacyUserId = (string) $row->{$colUser};
                $domainName = strtolower(trim((string) $row->{$colDomain}));
                if ($domainName === '') {
                    $bar->advance();
                    continue;
                }

                $user = User::query()->where('legacy_clientexec_id', $legacyUserId)->orWhere('id', $legacyUserId)->first();
                if (!$user) {
                    $bar->advance();
                    continue;
                }

                $exists = Domain::query()->where('user_id', $user->id)->where('domain', $domainName)->exists();
                if ($exists) {
                    $bar->advance();
                    continue;
                }

                if (!$dryRun) {
                    Domain::query()->create([
                        'user_id' => $user->id,
                        'domain' => $domainName,
                        'years' => 1,
                        'status' => $colStatus ? (string) ($row->{$colStatus} ?? 'imported') : 'imported',
                        'registrar' => 'imported',
                        'expires_at' => null,
                        'meta' => [
                            'imported_from' => 'clientexec',
                        ],
                    ]);
                }

                $bar->advance();
            }
        });

        $bar->finish();
        $this->newLine(2);
        $this->info('Domains import done.');
    }

    private function importInvoices(string $ce, bool $dryRun, int $chunk): void
    {
        if (!Schema::connection($ce)->hasTable('invoice')) {
            $this->line('--- Skipping invoices (ClientExec table "invoice" not found) ---');
            return;
        }

        $this->line('--- Importing invoices ---');

        $cols = Schema::connection($ce)->getColumnListing('invoice');
        $colId = $this->pickColumn($cols, ['id', 'invoiceid']);
        $colUser = $this->pickColumn($cols, ['userid', 'user_id', 'customerid', 'clientid']);
        $colStatus = $this->pickColumn($cols, ['status']);
        $colTotal = $this->pickColumn($cols, ['total', 'amount', 'balance']);
        $colDate = $this->pickColumn($cols, ['datedue', 'due_date', 'duedate', 'billdate', 'date']);

        if (!$colId || !$colUser) {
            $this->warn('Could not detect required columns in ClientExec invoice table.');
            return;
        }

        $query = DB::connection($ce)->table('invoice');
        $total = (clone $query)->count();
        $this->info("Found $total ClientExec invoice records.");

        $bar = $this->output->createProgressBar($total);
        $bar->start();

        $query->orderBy($colId)->chunk($chunk, function ($rows) use ($dryRun, $bar, $colId, $colUser, $colStatus, $colTotal, $colDate) {
            foreach ($rows as $row) {
                $legacyInvoiceId = (string) $row->{$colId};
                $legacyUserId = (string) $row->{$colUser};

                $user = User::query()->where('legacy_clientexec_id', $legacyUserId)->orWhere('id', $legacyUserId)->first();
                if (!$user) {
                    $bar->advance();
                    continue;
                }

                $number = 'CE-' . $legacyInvoiceId;

                if (Invoice::query()->where('number', $number)->exists()) {
                    $bar->advance();
                    continue;
                }

                $statusRaw = $colStatus ? (string) ($row->{$colStatus} ?? '') : '';
                $status = $this->mapInvoiceStatus($statusRaw);

                $totalVal = $colTotal ? (float) ($row->{$colTotal} ?? 0) : 0.0;
                if ($totalVal <= 0) {
                    // Some ClientExec schemas store totals in invoiceentry; keep but still import.
                    $totalVal = 0.0;
                }

                $dueDate = now()->toDateString();
                if ($colDate && !empty($row->{$colDate})) {
                    $dueDate = (string) $row->{$colDate};
                }

                if (!$dryRun) {
                    $inv = Invoice::query()->create([
                        'user_id' => $user->id,
                        'order_id' => null,
                        'number' => $number,
                        'status' => $status,
                        'due_date' => $dueDate,
                        'subtotal' => $totalVal,
                        'tax' => 0,
                        'total' => $totalVal,
                        'currency' => config('clubhoster.currency', 'PKR'),
                        'paid_at' => $status === 'paid' ? now() : null,
                        'meta' => [
                            'imported_from' => 'clientexec',
                            'legacy_invoice_id' => $legacyInvoiceId,
                        ],
                    ]);

                    // Optionally import invoice lines
                    if (Schema::connection($ce)->hasTable('invoiceentry')) {
                        $this->importInvoiceEntries($ce, $legacyInvoiceId, $inv->id);
                    }
                }

                $bar->advance();
            }
        });

        $bar->finish();
        $this->newLine(2);
        $this->info('Invoices import done.');
    }

    private function importInvoiceEntries(string $ce, string $legacyInvoiceId, int $newInvoiceId): void
    {
        try {
            $cols = Schema::connection($ce)->getColumnListing('invoiceentry');
            $colInv = $this->pickColumn($cols, ['invoiceid', 'invoice_id']);
            $colDesc = $this->pickColumn($cols, ['description', 'itemdescription', 'detail']);
            $colAmt = $this->pickColumn($cols, ['amount', 'price', 'total']);

            if (!$colInv || !$colAmt) {
                return;
            }

            $entries = DB::connection($ce)->table('invoiceentry')->where($colInv, $legacyInvoiceId)->get();
            foreach ($entries as $entry) {
                $desc = $colDesc ? trim((string) ($entry->{$colDesc} ?? 'ClientExec Item')) : 'ClientExec Item';
                $amt = (float) ($entry->{$colAmt} ?? 0);

                InvoiceItem::query()->create([
                    'invoice_id' => $newInvoiceId,
                    'description' => $desc,
                    'qty' => 1,
                    'unit_price' => $amt,
                    'total' => $amt,
                    'meta' => [
                        'imported_from' => 'clientexec',
                    ],
                ]);
            }
        } catch (\Throwable $e) {
            // ignore
        }
    }

    private function importTransactions(string $ce, bool $dryRun, int $chunk): void
    {
        if (!Schema::connection($ce)->hasTable('invoicetransaction')) {
            $this->line('--- Skipping transactions (ClientExec table "invoicetransaction" not found) ---');
            return;
        }

        $this->line('--- Importing transactions ---');

        $cols = Schema::connection($ce)->getColumnListing('invoicetransaction');
        $colId = $this->pickColumn($cols, ['id', 'transactionid', 'txn_id']);
        $colInvoice = $this->pickColumn($cols, ['invoiceid', 'invoice_id']);
        $colAmt = $this->pickColumn($cols, ['amount', 'total']);
        $colGateway = $this->pickColumn($cols, ['gateway', 'paymentmethod', 'payment_method']);
        $colRef = $this->pickColumn($cols, ['reference', 'transactionref', 'txnref', 'transid']);

        if (!$colInvoice) {
            $this->warn('Could not detect required columns in ClientExec invoicetransaction table.');
            return;
        }

        $query = DB::connection($ce)->table('invoicetransaction');
        $total = (clone $query)->count();
        $this->info("Found $total ClientExec transaction records.");

        $bar = $this->output->createProgressBar($total);
        $bar->start();

        $query->orderBy($colId ?: $colInvoice)->chunk($chunk, function ($rows) use ($dryRun, $bar, $colInvoice, $colAmt, $colGateway, $colRef) {
            foreach ($rows as $row) {
                $legacyInvoiceId = (string) $row->{$colInvoice};
                $invoice = Invoice::query()->where('meta->legacy_invoice_id', $legacyInvoiceId)->first();
                if (!$invoice) {
                    $bar->advance();
                    continue;
                }

                $amount = $colAmt ? (float) ($row->{$colAmt} ?? 0) : 0.0;
                $gateway = $colGateway ? (string) ($row->{$colGateway} ?? 'imported') : 'imported';
                $ref = $colRef ? (string) ($row->{$colRef} ?? ('CE-TXN-' . $legacyInvoiceId)) : ('CE-TXN-' . $legacyInvoiceId);

                $exists = Transaction::query()->where('invoice_id', $invoice->id)->where('txn_ref', $ref)->exists();
                if ($exists) {
                    $bar->advance();
                    continue;
                }

                if (!$dryRun) {
                    Transaction::query()->create([
                        'invoice_id' => $invoice->id,
                        'user_id' => $invoice->user_id,
                        'gateway' => 'clientexec_' . Str::slug($gateway),
                        'txn_ref' => $ref,
                        'amount' => $amount,
                        'currency' => $invoice->currency,
                        'status' => 'imported',
                        'response_code' => null,
                        'response_message' => 'Imported from ClientExec',
                        'raw_request' => null,
                        'raw_response' => null,
                    ]);
                }

                $bar->advance();
            }
        });

        $bar->finish();
        $this->newLine(2);
        $this->info('Transactions import done.');
    }

    private function pickColumn(array $columns, array $candidates): ?string
    {
        $columnsLower = array_map('strtolower', $columns);
        foreach ($candidates as $candidate) {
            $candidate = strtolower($candidate);
            $idx = array_search($candidate, $columnsLower, true);
            if ($idx !== false) {
                return $columns[$idx];
            }
        }
        return null;
    }

    private function mapInvoiceStatus(string $raw): string
    {
        $raw = strtolower(trim($raw));

        // ClientExec often uses numeric codes; keep a safe mapping.
        if ($raw === '1' || $raw === 'paid' || $raw === 'p') {
            return 'paid';
        }
        if ($raw === '0' || $raw === 'unpaid' || $raw === 'u') {
            return 'unpaid';
        }
        if (str_contains($raw, 'cancel')) {
            return 'cancelled';
        }

        return $raw !== '' ? $raw : 'unpaid';
    }
}
